#include <Wire.h>
#include <LiquidCrystal_I2C.h>

// Initialize I2C LCD (use the detected address, usually 0x27 or 0x3F)
LiquidCrystal_I2C lcd(0x27, 16, 2); // Address, columns, rows

// Pin definitions for sensors
const int lightSensorPin = A0;
const int tempSensorPin = A1;
const int distanceTriggerPin = 3;
const int distanceEchoPin = 2;

// RGB LED pins
const int rgbRedPin = 6;
const int rgbGreenPin = 5;
const int rgbBluePin = 4;

// Reset button pin
const int resetButtonPin = 7;

// Target values and tolerances
const int targetLight = 0; // Approximate intensity matching a phone's flashlight
const int lightTolerance = 5;
const float targetTemp = 24.0; // Average finger tip temperature
const float tempTolerance = 2.0;
const float targetDistance = 8.0;
const float distanceTolerance = 1.0;

// Task completion flags
bool tasksCompleted[3] = {false, false, false};

void setup() {
  // Initialize LCD
  lcd.init();        // Initialize the LCD
  lcd.backlight();   // Turn on the LCD backlight

  // Initialize RGB LED pins
  pinMode(rgbRedPin, OUTPUT);
  pinMode(rgbGreenPin, OUTPUT);
  pinMode(rgbBluePin, OUTPUT);

  // Initialize distance sensor pins
  pinMode(distanceTriggerPin, OUTPUT);
  pinMode(distanceEchoPin, INPUT);

  // Initialize reset button pin
  pinMode(resetButtonPin, INPUT_PULLUP);

  // Start message
  lcd.print("Awaken the");
  lcd.setCursor(0, 1);
  lcd.print("Guardian...");
  delay(5000);
  lcd.clear();
}

void loop() {
  // Ensure tasks run only if they are incomplete
  if (!tasksCompleted[0] && checkLightSensor()) {
    tasksCompleted[0] = true;
    lcd.clear();
    lcd.print("Eye of Light");
    lcd.setCursor(0, 1);
    lcd.print("Activated!");
    activateRGB(255, 255, 255); // White light
    delay(5000);
    deactivateRGB();
  } else if (!tasksCompleted[0]) {
    lcd.clear();
    lcd.print("Light incorrect");
    lcd.setCursor(0, 1);
    lcd.print("Adjust light!");
    delay(1000);
  }

  if (!tasksCompleted[1] && checkTempSensor()) {
    tasksCompleted[1] = true;
    lcd.clear();
    lcd.print("Eye of Fire");
    lcd.setCursor(0, 1);
    lcd.print("Activated!");
    activateRGB(255, 0, 0); // Red light
    delay(5000);
    deactivateRGB();
  } else if (!tasksCompleted[1]) {
    lcd.clear();
    lcd.print("Temp incorrect");
    lcd.setCursor(0, 1);
    lcd.print("Adjust temp!");
    delay(1000);
  }

  if (!tasksCompleted[2] && checkDistanceSensor()) {
    tasksCompleted[2] = true;
    lcd.clear();
    lcd.print("Eye of Space");
    lcd.setCursor(0, 1);
    lcd.print("Activated!");
    activateRGB(0, 0, 255); // Blue light
    delay(5000);
    deactivateRGB();
  } else if (!tasksCompleted[2]) {
    lcd.clear();
    lcd.print("Distance wrong");
    lcd.setCursor(0, 1);
    lcd.print("Adjust pos!");
    delay(1000);
  }

  // Check if all tasks are completed
  if (tasksCompleted[0] && tasksCompleted[1] && tasksCompleted[2]) {
    lcd.clear();
    lcd.print("Guardian's");
    lcd.setCursor(0, 1);
    lcd.print("Secret Revealed!");
    victorySequence();
    delay(10000);

    // Wait for reset button to be pressed
    while (true) {
      if (digitalRead(resetButtonPin) == HIGH) {
        resetSystem();
        break;
      }
    }
  }
}

// Light sensor check
bool checkLightSensor() {
  int lightValue = analogRead(lightSensorPin);
  return abs(lightValue - targetLight) <= lightTolerance;
}

// Temperature sensor check
bool checkTempSensor() {
  int tempValue = analogRead(tempSensorPin);
  float voltage = tempValue * 5.0 / 1023.0;
  float tempC = (voltage - 0.5) * 100;
  return abs(tempC - targetTemp) <= tempTolerance;
}

// Distance sensor check
bool checkDistanceSensor() {
  digitalWrite(distanceTriggerPin, LOW);
  delayMicroseconds(2);
  digitalWrite(distanceTriggerPin, HIGH);
  delayMicroseconds(10);
  digitalWrite(distanceTriggerPin, LOW);

  long duration = pulseIn(distanceEchoPin, HIGH);
  float distance = duration * 0.034 / 2; // Convert to cm
  return abs(distance - targetDistance) <= distanceTolerance;
}

// RGB LED control
void activateRGB(int red, int green, int blue) {
  analogWrite(rgbRedPin, red);
  analogWrite(rgbGreenPin, green);
  analogWrite(rgbBluePin, blue);
}

void deactivateRGB() {
  analogWrite(rgbRedPin, 0);
  analogWrite(rgbGreenPin, 0);
  analogWrite(rgbBluePin, 0);
}

// Victory sequence for RGB LED
void victorySequence() {
  for (int i = 0; i < 3; i++) {
    activateRGB(0, 255, 0); // Green light
    delay(500);
    deactivateRGB();
    delay(500);
  }
}

// Reset system to initial state
void resetSystem() {
  // Reset task completion flags
  for (int i = 0; i < 3; i++) {
    tasksCompleted[i] = false;
  }

  // Clear LCD and show start message
  lcd.clear();
  lcd.print("Awaken the");
  lcd.setCursor(0, 1);
  lcd.print("Guardian...");
  delay(5000);
  lcd.clear();

  // Deactivate RGB LED
  deactivateRGB();
}